#
# basic function definitions
#

basiclists := proc(d::nonnegint)
options remember;
local i;
  [ seq([ 0$(i-1), 1, 0$(d-i) ], i = 1..d) ]
end;

# internal functions: no type checking for better performance

Dotprod := proc(v, w)
local i;
#  `+`(op(zip(`*`, v, w))) # zip is not built-in!
  `+`(seq(v[i]*w[i], i = 1..nops(v)))
end;

Dotprodwith := proc(v)
local i, w;
  unapply(`+`(seq(v[i]*w[i], i = 1..nops(v))), w)
end;

# this function is for the user

CONVEX['dotprod'] := proc(v::ray, w::ray)
local i, n;
  n := &?(type(v, list), nops(v),
       &?(type(v, Vector), op(Vector_OP_nrows, v),
	                   op(vector_OP_nrows, eval(v))));
  if n <> &?(type(w, list), nops(w),
	  &?(type(w, Vector), op(Vector_OP_nrows, w),
			      op(vector_OP_nrows, eval(w))))
    then ERRORnotsamespace
    else `+`(seq(v[i]*w[i], i = 1..n))
  fi
end;

transpose := proc(A::{[], listlist, matrix, Matrix}, ncols::nonnegint)
# ncols is only used if A = []
local m, i, j;
  if A = []
    then m := ncols
    else m := mat_ncols(A)
  fi;
  [seq([seq(A[i, j], i = 1..mat_nrows(A))], j = 1..m)]
end;

inverse := proc(A_)
local A, B, n, i, j, v, w;
  if type(A, list)
    then A := A_
    else A := convert(A_, listlist)
  fi;
  n := nops(A);
  if n = 0
    then RETURN([])
  elif n <> nops(A[1])
    then RETURN(FAIL)
  fi;
  B := basiclists(n);
  for i to n do
    for j from i to n while A[j, i] = 0 do od;
    if j > n then RETURN(FAIL) fi;
    v := A[j];
    w := B[j]/v[i];
    v := v/v[i];
    A[j] := A[i]; A[i] := v;
    B[j] := B[i]; B[i] := w;
    for j to n do
      if j <> i
        then
          B[j] := B[j]-A[j, i]*w;
          A[j] := A[j]-A[j, i]*v;
      fi
    od
  od;
  B
end;

matvecmul := proc(A, v)
  map(Dotprodwith(v), A)
end;

matmatmul := proc(A, B)
# the ROWS of the result are the ROWS of B multiplied by A
  map2(matvecmul, A, B)
end;

reducevec := proc(v)
local n;
  n := abs_igcd(igcd(op(v)));
  if n = 0  # really necessary?
    then v
    else v/n
  fi
end;

reduceqvec := proc(v)
local n, vl;
  if type(v, list) # faster than converting also for type list
    then vl := v
    else vl := convert(v, list)
  fi;
  n := abs_igcd(igcd(op(map(numer, vl))));
  if n = 0
    then vl
    else (ilcm(op(map(denom, vl)))/n)*vl
  fi
end;

combinevecs := proc(v, w)
  reducevec(w[2]*v[1]-v[2]*w[1])
end;

gausselim := proc(A::list, A2::list)
# reduces and returns A
# if A2 is given, it is also reduced (with the help of A) and returned
local B, B2, i, j, k, m, n, s;
  m := nops(A);
  n := &?(m = 0, 0, nops(A[1]));
    # If A = [], then n = 0 will immediately terminate the loop.
    # Note that we need to apply "reduceqvec" to A2 in this case!
  B := A;
  if nargs = 2 then B2 := A2 fi;
  j := 0;
  for i do
    for j from j+1 to n do
      for k from i to m while B[k, j] = 0 do od;
      if k <= m
        then
          B := subsop(k = B[i], i = B[k]/B[k, j], B);
          break
      fi
    od;
    if j <= n
      then s := B[i]
    elif nargs = 2
      then RETURN(map(reduceqvec, B[1..i-1]), map(reduceqvec, B2))
      else RETURN(map(reduceqvec, B[1..i-1]))
    fi;
    B := subsop(i = B[i], map(proc(v) v-v[j]*s end, B));
    if nargs = 2
      then B2 := map(proc(v) v-v[j]*s end, B2)
    fi
  od
end;

gausselim2 := proc(A::list, A2::list)
# reduces A2 with the help of A
local B, B2, s, j;
  B := A;
  B2 := A2;
  j := 0;
  for s in B do
    for j from j+1 while s[j] = 0 do od;
    s := s/s[j];
    B2 := map(proc(v) v-v[j]*s end, B2)
  od;
  map(reduceqvec, B2)
end;

# issamesubspace := proc(l1::list, l2::list)
# # l1, l2 are lists of linear independent vectors
# # this functions returns true, if they span the same space
# local nl1;
#   nl1 := nops(l1);
#   l1 = l2 or (nl1 = nops(l2) and linrank([op(l1), op(l2)]) = nl1)
#   # note that linrank is called with a list with at least two entries
# end;

det := proc(A_::list)
local d, n, A, i, j, v;
  if A_ = [] then RETURN(1) fi;
  d := 1;
  A := A_;
  n := nops(A);
  if nops(A[1]) <> n then ERROR("not a square matrix") fi;
  for i to n do
    for j from i to n while A[j, i] = 0 do od;
    if j > n
      then RETURN(0)
    elif j <> i
      then d := -d; A := subsop(i = A[j], j = A[i], A)
    fi;
    v := A[i]/A[i, i];
    d := d*A[i, i];
    for j from j+1 to n do
      A[j] := A[j]-A[j, i]*v
    od
  od;
  d
end;

hermite1 := proc(B::list, full::boolean)
# if full = true, computes the Hermite normal form of B
# otherwise, computes an "upper triangular" form
local A, i0, i, j, m, d, a1, a2, c1, c2;
  if B = [] then RETURN([]) fi;
  A := B;
  m := nops(A);
  i0 := 1;
  for j to nops(A[1]) while i0 <= m do
    a1 := A[i0, j];
    for i from i0+1 to m do
      a2 := A[i, j];
      if a2 = 0 then next fi;
      d := abs_igcd(igcdex(a1, a2, 'c1', 'c2'));
      a1 := a1/d; a2 := a2/d;
      A := subsop(i0 = c2*A[i]+c1*A[i0], i = a1*A[i]-a2*A[i0], A);
      a1 := d
    od;
    if a1 = 0
      then next
    elif a1 < 0
      then a1 := -a1; A[i0] := -A[i0]
    fi;
    if full
      then
	for i to j-1 do
#	  a2 := floor(A[i, j]/a1);
	  A := subsop(i = A[i]-floor(A[i, j]/a1)*A[i0], A);
	od
    fi;
    i0 := i0+1
  od;
  A
end;

# hermite2 := proc(B::list, full::boolean, U_::evaln)
# # like hermite1, but with an additional matrix for change of coordinates:
# #   A := hermite(B, true, U);
# # is equivalent to
# #   A := ihermite(B, 'U');
# local A, U, i0, i, j, m, d, a1, a2, c1, c2;
#   if B = [] then U_ := []; RETURN([]) fi;
#   A := B;
#   m := nops(A);
#   U := basiclists(m);
#   i0 := 1;
#   for j to nops(A[1]) while i0 <= m do
#     a1 := A[i0, j];
#     for i from i0+1 to m do
#       a2 := A[i, j];
#       if a2 = 0 then next fi;
#       d := abs_igcd(igcdex(a1, a2, 'c1', 'c2'));
#       a1 := a1/d; a2 := a2/d;
#       A := subsop(i0 = c2*A[i]+c1*A[i0], i = a1*A[i]-a2*A[i0], A);
#       U := subsop(i0 = c2*U[i]+c1*U[i0], i = a1*U[i]-a2*U[i0], U);
#       a1 := d
#     od;
#     if a1 = 0
#       then next
#     elif a1 < 0
#       then a1 := -a1; A[i0] := -A[i0]; U[i0] := -U[i0]
#     fi;
#     if full
#       then
# 	    for i to j-1 do
# 	      a2 := floor(A[i, j]/a1);
# 	      A := subsop(i = A[i]-a2*A[i0], A);
# 	      U[i] := U[i]-a2*U[i0]
# 	    od
#     fi;
#     i0 := i0+1
#   od;
#   U_ := U;
#   A
# end;

hermite2inv := proc(B::list, full::boolean, V_::evaln)
# like hermite1, but with an additional matrix for change of coordinates:
#   A := hermite(B, true, V);
# is equivalent to
#   A := ihermite(B, 'U');
#   V := transpose(inverse(U));
# without having U.
# Hence, if B is an m x n matrix of rank r, then the ROWS of V are a basis
# of Z^m such that the first r are a basis of the saturated subspace spanned
# by the COLUMNS of B. (But be careful with the case B = [] !)
local A, V, i0, i, j, m, d, a1, a2, c1, c2;
  if B = [] then V_ := []; RETURN([]) fi;
  A := B;
  m := nops(A);
  V := basiclists(m);
  i0 := 1;
  for j to nops(A[1]) while i0 <= m do
    a1 := A[i0, j];
    for i from i0+1 to m do
      a2 := A[i, j];
      if a2 = 0 then next fi;
      d := abs_igcd(igcdex(a1, a2, 'c1', 'c2'));
      a1 := a1/d; a2 := a2/d;
      A := subsop(i0 = c2*A[i]+c1*A[i0], i = a1*A[i]-a2*A[i0], A);
      V := subsop(i0 = a2*V[i]+a1*V[i0], i = c1*V[i]-c2*V[i0], V);
      a1 := d
    od;
    if a1 = 0
      then next
    elif a1 < 0
      then a1 := -a1; A[i0] := -A[i0]; V[i0] := -V[i0]
    fi;
    if full
      then
	for i to j-1 do
	  a2 := floor(A[i, j]/a1);
	  A := subsop(i = A[i]-a2*A[i0], A);
	  V[i0] := a2*V[i]+V[i0]
	od
    fi;
    i0 := i0+1
  od;
  V_ := V;
  A
end;

hermite3 := proc(B::list, full::boolean, U_::evaln, V_::evaln)
# like hermite1, but with two additional matrices for change of coordinates:
#   A := hermite(B, true, U, V);
# is equivalent to
#   A := ihermite(B, 'U');
#   V := transpose(inverse(U));
# with the value of U still available.
local A, U, V, i0, i, j, m, d, a1, a2, c1, c2;
  if B = [] then U_ := []; V_ := []; RETURN([]) fi;
  A := B;
  m := nops(A);
  V := basiclists(m);
  U := V;
  i0 := 1;
  for j to nops(A[1]) while i0 <= m do
    a1 := A[i0, j];
    for i from i0+1 to m do
      a2 := A[i, j];
      if a2 = 0 then next fi;
      d := abs_igcd(igcdex(a1, a2, 'c1', 'c2'));
      a1 := a1/d; a2 := a2/d;
      A := subsop(i0 = c2*A[i]+c1*A[i0], i = a1*A[i]-a2*A[i0], A);
      U := subsop(i0 = c2*U[i]+c1*U[i0], i = a1*U[i]-a2*U[i0], U);
      V := subsop(i0 = a2*V[i]+a1*V[i0], i = c1*V[i]-c2*V[i0], V);
      a1 := d
    od;
    if a1 = 0
      then next
    elif a1 < 0
      then a1 := -a1; A[i0] := -A[i0]; U[i0] := -U[i0]; V[i0] := -V[i0]
    fi;
    if full
      then
	for i to j-1 do
	  a2 := floor(A[i, j]/a1);
	  A := subsop(i = A[i]-a2*A[i0], A);
	  U[i] := U[i]-a2*U[i0];
	  V[i0] := a2*V[i]+V[i0]
	od
    fi;
    i0 := i0+1
  od;
  U_ := U; V_ := V;
  A
end;

latticebasis := proc(L::{listlist, []})
# returns a lattice basis of the saturation of the subspace spanned by the
# rows of L. The rows of L must be linear independent!
local V;
  hermite2inv(transpose(L, 0), false, V);
    # if L = [] then the result will be [] anyway
  V[1..nops(L)]
end;

# routines for making lists with lots of zeroes

embed := proc(n::nonnegint, L1::list, i1::nonnegint, L2::list, i2::nonnegint)
# returns a list of length n containing
# the list L1 from pos i1+1 on, and (if given)
# the list L2 from pos i2+1 on, rest zeroes
# i1 and i2 must not be equal!
  if nargs = 3
    then [ 0$i1, op(L1), 0$n-i1-nops(L1) ]
  elif i1 < i2
    then [ 0$i1, op(L1), 0$i2-i1-nops(L1), op(L2), 0$n-i2-nops(L2) ]
    else [ 0$i2, op(L2), 0$i1-i2-nops(L2), op(L1), 0$n-i1-nops(L1) ]
  fi
end;

#embed_ll := proc(n::nonnegint, L::{listlist, []})
#local i, l, S;
#  S := NULL;
#  i := 0;
#  for l in L do
#    S := S, embed(d, l, i);
#    i := i+nops(l)
#  od;
#  [ S ]
#end;

# routines for transformation affine <-> homogeneous

macro(X0 = -1); # position of additional coordinate
macro(X_1 = -2); # position of last real coordinate

affine := proc(v::list)
local a;
  a := v[-1];
  if a = 0
    then v[1..-2]
    else v[1..-2]/a
  fi
end;

homog := proc(v::ray, c::rational)
  reduceqvec([op(convert(v, list)), c])
end;

homog_1 := proc(v::list, v_1::rational, c::rational)
  reduceqvec([op(v), v_1, c])
end;

dehomog := proc(h::ray)
  [ h[1..-2], h[-1] ]
end;

mat_homog := proc(A::listlist, v::list)
local i;
  [seq([op(A[i]), v[i]], i = 1..nops(A)), origin(nops(A[1])+1)]
end;

origin := proc(d::posint)
options remember;
  [ 0$d-1, 1 ]
end;

Isaffine := proc(r::ray)
  evalb(r[-1] > 0)
end;

affh2ray := proc(h::{affhplane, affhspace})
  if type(op(1, h), rational)
    then  homog(op(2, h), -op(1, h))
    else -homog(op(1, h), -op(2, h))
  fi
end;

#
# shift array by -1
#

shiftarray := proc(A)
local i, i0, i1, B;
  i0 := op(vector_OP_from, eval(A));
  i1 := op(vector_OP_to, eval(A));
  B := array(i0-1..i1-1);
  for i from i0 to i1 do
    B[i-1] := A[i]
  od;
  eval(B)
end;
