#
# Fourier-Motzkin elimination
#

maximalsets := proc(l::list(set), min::integer, max::integer)
# Returns a list containing the position of the maximal elements of l
# with between min and max elements. Of two equal sets only the first is
# considered.
# If min and max are missing, they are 0 and infinity, repsectively.
local nl, i, m;
  nl := [ seq([i, nops(l[i])], i = 1..nops(l)) ];
  if nargs = 3
    then nl := remove(proc(x) x[2] < min or x[2] > max end, nl)
         # remove((x, y, z) -> evalb(x[2] < y or x[2] > z), nl, min, max)
  fi;
  nl := map(x -> x[1], sort(nl, cmpop2));
  m := [];
  for i in nl do
    if not ormap(proc(x) l[i] subset l[x] end, m)
           # exists((x, y, z) -> evalb(z subset y[x]), m, dummy, l, l[i])
      then m := [ op(m), i ]
    fi
  od;
  m
end;

motzkin := proc(C::{CONE, POLYHEDRON}, neweq::list, newineq::list)
# returns a true cone whose lines and rays are gausselim'ed
local d, l0, l, r, sat, oldineq,
    newc, ci, ineqno, nooldineq, sl, sr, sw, i, j, k,
    L,
# adtest,
   face2d, adjacent, rpos, rneg, r0, addr, addsat, csat,
   dpw_newc_ci;

  # during the reduction process, d counts the codimension of
  # linear independent equations (which are relevant for rays)
  d := Dim(C);
  l0 := Lines(C);
  l := l0;
  r := Rays(C);
  sat := Incidentfacets(C);
  oldineq := Hspaces(C);
  nooldineq := nops(oldineq);

  newc := [op(neweq), op(newineq)];
  userinfo(2, CONVEX, nops(newc), `constraints`);

  for ci to nops(newc) while nops(l) > 0 or nops(r) > 0 do
    ineqno := nooldineq+ci-nops(neweq);

    userinfo(4, CONVEX, nops(l), `lines,`, nops(r), `rays`);
    userinfo(3, CONVEX, `constraint #`, ci);

#    sl := map((x,y) -> [x, Dotprod(x,y)] , l, newc[ci]);
#    sr := map((x,y) -> [x, Dotprod(x,y)] , r, newc[ci]);
    dpw_newc_ci := Dotprodwith(newc[ci]);
    sl := map(proc(v) [v, dpw_newc_ci(v)] end, l);
    sr := map(proc(v) [v, dpw_newc_ci(v)] end, r);

    if exists(x -> x[2] <> 0, sl, i)
      then
        userinfo(4, CONVEX, `reducing lineality space`);

        sw := sign(sl[i, 2])*sl[i];
        sl := subsop(i = NULL, sl);

        l := map(combinevecs, sl, sw);
        r := map(combinevecs, sr, sw);

	if ineqno > nooldineq
	  then # processing inequaltity
	    r := [op(r), sw[1]];
	    sat := [op(map(`union`, sat, { ineqno })), { $1..ineqno-1 }]
          else # processing equation
            d := d-1;
 	fi
      else
        userinfo(4, CONVEX, `computing new rays ...`);

#	rpos := []; rneg := []; r0 := [];
#	for i to nops(r) do
#	  if sr[i, 2] = 0
#	    then r0 := [op(r0), i]
#	  elif sr[i, 2] > 0
#	    then rpos := [op(rpos), i]
#	    else rneg := [op(rneg), i]
#          fi
#	od;

        L :=  [$1..nops(r)];
        rpos, rneg := selectremove(proc(x) sr[x, 2] > 0 end, L);
        r0, rneg := selectremove(proc(x) sr[x, 2] = 0 end, rneg);

#    L := [$1..nops(r)];
#    r0 := select((x,y) -> y[x, 2] = 0, L, sr);
#    rpos := select((x, y) -> y[x, 2] > 0, L, sr);
#    rneg := select((x, y) -> y[x, 2] < 0, L, sr);

        userinfo(5, CONVEX, `r0 = `, nops(r0),
                 `r+ = `, nops(rpos), `r- = `, nops(rneg));

	addr := []; addsat := [];
        face2d := d-nops(l)-2;
# adtest := 0
	for j in rneg do
	  for i in rpos do
	    csat := sat[i] intersect sat[j];
	    if nops(csat) < face2d then next fi;
# adtest := adtest+1;
	    adjacent := true;
	    for k to nops(r) do
	      if csat subset sat[k] and k <> j and k <> i
		then adjacent := false; break
	      fi
	    od;
	    if adjacent
              then
	        addr := [op(addr), combinevecs(sr[j], sr[i]) ];
	        addsat := [op(addsat), csat]
            fi
	  od
	od;
# lprint(adtest, `adjacency tests out of`, nops(rpos)*nops(rneg));
#        userinfo(4, CONVEX, nops(addr),`rays added`);

	if ineqno > nooldineq
	  then # processing inequality
	    for i in r0 do
	      sat := subsop(i = sat[i] union { ineqno }, sat)
	    od;
	    addsat := map(`union`, addsat, { ineqno });
	    r0 := [op(r0), op(rpos)]
          # else: processing equation
        elif nops(rpos)+nops(rneg) > 0 # is equation non-redundant?
          then
            d := d-1
	fi;
	r := [ seq(r[i], i = r0), op(addr) ];
	sat := [ seq(sat[i], i = r0), op(addsat) ]
    fi
  od;

  if l <> l0
    then
      userinfo(3, CONVEX, `normalising lines and rays`);
      l, r := gausselim(l, r)
  fi;

  # Incidentrays(C) is still valid if no ineqs were added.
  # This is used in pointedcone. OBSOLETE!!!
  # used also in linearhull
  NEWCONE(Ambientdim(C), l, r, sat,
    [op(Hplanes(C)), op(neweq)], [op(oldineq), op(newineq)], Incidentrays(C));
end;

transposesat := proc(sat::list, entries::nonnegint)
local sat2, i, j;
  sat2 := [ {} $ entries  ];
  for i to nops(sat) do
    for j in sat[i] do
      sat2 := subsop(j = sat2[j] union { i }, sat2)
    od
  od;
  sat2
end;

reduce := proc(C::CONE)
# lines and rays of C must be gausselim'ed
# hplanes and hspaces are gausselim'ed
local d, l, r, sat, i, j, sat2, eq, ineq, nr;

  d := Ambientdim(C);
  l := Lines(C);
  r := Rays(C);

  if nops(l) = 0 and nops(r) = 0 then RETURN(zerocone(d)) fi;

  sat := Incidentfacets(C);
  eq := Hplanes(C);
  ineq := Hspaces(C);

  userinfo(3, CONVEX, `transposing ray-facet incidence table`);
  sat2 := transposesat(sat, nops(ineq));

  userinfo(3, CONVEX, `detecting implicit equations`);
  nr := nops(r);
  eq := [op(eq),
	 seq(ineq[i], i = select(proc(x) nops(sat2[x]) = nr end,
                                 [$1..nops(ineq)]))];
# [ op(eq),
#   seq(ineq[i], i = select((x, y, z) -> evalb(nops(y[x]) = z),
#                           [ $1..nops(ineq) ], sat2, nops(r))) ];

  userinfo(3, CONVEX, `reducing and normalising equations`);
  eq := gausselim(eq);

  userinfo(3, CONVEX, `reducing and normalising inequalities`);
  j := maximalsets(sat2, d-nops(eq)-nops(l)-1, nops(r)-1);
    # d-nops(eq)-nops(l)-1 is the minimal number of rays in a facet

  if nops(j) = nops(sat2)
    then # no redundancies
      userinfo(2, CONVEX, `all inequalities induce facets`);
      ineq := gausselim2(eq, ineq);
    else
      ineq := gausselim2(eq, [seq(ineq[i], i = j)]);
      sat2 := [ seq(sat2[i], i = j) ];

      userinfo(3, CONVEX, `transposing facet-ray incidence table`);
      sat := transposesat(sat2, nops(r))
  fi;

  NEWCONE(d, l, r, sat, eq, ineq, sat2)
end;

#
# functions for intersection/hull computations
#

Intersection := proc(n::nonnegint, Cl::list, eqs::list, ineqs::list)
# eqs and ineqs must be reduced!
# always returns a true cone
local Cl2, eqs2, ineqs2, C, i, NC;
  if Cl = []
    then
      C := fullcone(n);
      eqs2 := eqs;
      ineqs2 := ineqs
    else
      i := findmax(CONE['codim'], Cl);
      C := Cl[i];
      Cl2 := subsop(i = NULL, Cl);
      eqs2 := [ seq(op(Hplanes(i)), i = Cl2), op(eqs) ];
      ineqs2 := [ seq(op(Hspaces(i)), i = Cl2), op(ineqs) ];
  fi;

  userinfo(2, CONVEX, '`starting with`',
           OBJ_TYPE(C)['_print'](op(2..-1, C)));
  NC := motzkin(C, eqs2, ineqs2);
  if Rays(NC) = Rays(C) and Lines(NC) = Lines(C)
     # nothing has changed, but NC may contain redundant constraints
    then CONE['`convert/CONE`'](C) # remove additional entries
    else reduce(NC)
  fi
end;

Hull := proc(n::nonnegint, Cl, ll, rl)
# ll and rl must be reduced!
# always returns a true cone
  Dual(Intersection(n, map(Dual, Cl), ll, rl))
end;
