#
# POLYHEDRA
#

`type/POLYHEDRON` := proc(x)
  op(0, x) = OBJ and OBJ_TYPE(x)[BASE] = 'POLYHEDRON'
end;

POLYHEDRON[_print] := proc(d::posint, l::list, r::list, sat::list,
                  eq::list, ineq::list, sat2::list, nr::integer)
  if nr = 0 and l = []
    then 'POLYTOPE'(d-1, d-nops(eq)-1, nops(r), nops(ineq))
    else 'POLYHEDRON'(d-1, d-nops(eq)-1, nops(l),
           [nops(r)-abs(nr), abs(nr)],
           &?(nr > 0, nops(ineq), [nops(ineq)-1]))
  fi
end;

POLYHEDRON[BASE] := 'POLYHEDRON';
POLYHEDRON[EL] := 'PFACE';
POLYHEDRON[POSDOM] := 'PCOMPLEX'; # possible domain

`type/POLYHEDRON1` := proc(x)
  type(x, POLYHEDRON) and Rays(x) <> []
end;

`type/POLYHEDRON0` := proc(x)
# contains 0
  type(x, POLYHEDRON) and CONE['contains'](x, origin(Ambientdim(x)))
end;

`type/POLYTOPE` := proc(x)
  type(x, POLYHEDRON) and Isbounded(x)
end;

# `type/POLYTOPE1` := proc(x)
#   type(x, POLYTOPE) and Rays(x) <> []
# end;

# `type/POLYTOPE0` := proc(x)
#   type(x, POLYHEDRON0) and Isbounded(x)
# end;

`type/POLYHEDRON_0` := proc(x)
# contains 0 in the relative interior
  type(x, POLYHEDRON) and CONE['containsrelint'](x, origin(Ambientdim(x)))
end;

`type/POLYHEDRON__0` := proc(x)
# contains 0 in the interior
  type(x, POLYHEDRON_0) and Hplanes(x) = []
end;

# `type/AFFINECONE` := proc(x)
#   type(x, POLYHEDRON) and nops(Rays(x))+NRays(x) = 1
#   # equality can never hold if NRays is positive
# end;

NEWPOLYHEDRON := proc(P::{CONE, POLYHEDRON})
# sorts the rays, vertices and facets of a polyhedron
# The true type of P does not matter, but contains(origin(d), P) must be true.
# If necessary, P is reduced to the empty polyhedron.
local r, sat, ri, vi, rnex, h, sat2, nh;
  # sorting rays and vertices. The rays come first
  r := Rays(P);
  sat := Incidentfacets(P);
  ri := 0; vi := nops(r)+1;
  rnex := NULL;
  do
    for ri from ri+1 to vi-1 while not Isaffine(r[ri]) do od;
    for vi from vi-1 to ri by -1 while Isaffine(r[vi]) do od;
    if vi <= ri then break fi;
    r := subsop(vi = r[ri], ri = r[vi], r);
    sat := subsop(vi = sat[ri], ri = sat[vi], sat);
    rnex := rnex, vi = ri, ri = vi
  od;
  ASSERT(ri = vi+1, "rays and vertices incorrectly sorted");
  h := Hspaces(P);
  sat2 := subs([rnex], Incidentrays(P));
  if vi = nops(r)
    then
      # userinfo(3, POLYHEDRON, '`reducing to empty polyhedron`');
      RETURN(emptypolyhedron(Ambientdim(P)-1))
  elif POLYHEDRON['dim'](P) <= vi+nops(Lines(P))
    then # possibly a facet at infinity
      # sorting hspaces, the "hspace at infinity" (if exists) comes at the end
      # (the empty facet does not fall into this category)
      nh := nops(h);
      for ri to nh do
        if not raynos_isaffine(sat2[ri], vi)
          then
#            if ri < nh then
            h := subsop(ri = h[nh], nh = h[ri], h);
            sat2 := subsop(ri = sat2[nh], nh = sat2[ri], sat2);
            sat := subs([ri = nh, nh = ri], sat);
#            fi;
            vi := -vi;
            break
        fi
      od
  fi;
  OBJ('POLYHEDRON', Ambientdim(P), Lines(P), r, sat, Hplanes(P), h, sat2, vi)
end;

# basic polyhedra

emptypolyhedron := proc(d::nonnegint)
options remember;
  if nargs > 1
    then ERRORnargs
    else subsop(OBJ_OP_TYPE = 'POLYHEDRON', OBJ(op(zerocone(d+1)), 0))
  fi
end;

CONVEX[emptypolytope] := eval(emptypolyhedron);

fullpolyhedron := proc(d::nonnegint)
options remember;
local b;
  if nargs > 1 then ERRORnargs fi;
  b := basiclists(d+1);
  OBJ('POLYHEDRON', d+1, b[1..d], [b[-1]], [{}], [], [b[-1]], [{}], 0)
end;

# simple operations

POLYHEDRON[lines] := proc(P)
  if nargs > 1
    then ERRORnargs
  elif Rays(P) = []
    then FAIL
    else map(affine, Lines(P))
  fi
end;

Vertices := proc(P)
  if nargs > 1
    then ERRORnargs
    else map(affine, Rays(P)[abs(NRays(P))+1..-1])
  fi
end;

POLYHEDRON[rays] := proc(P)
  if nargs > 1
    then ERRORnargs
    else map(affine, Rays(P)[1..abs(NRays(P))])
  fi
end;

POLYHEDRON[hplanes] := proc(P)
  if nargs > 1
    then ERRORnargs
  elif POLYHEDRON['isempty'](P)
    then [[0$POLYHEDRON['ambientdim'](P)] = 1]
    else map(x -> x[1] = -x[2], map(dehomog, Hplanes(P)))
  fi
end;

POLYHEDRON[hspaces] := proc(P)
  if nargs > 1
    then ERRORnargs
  elif POLYHEDRON['isaffine'](P)
    then [] # suppress the trivial inequality "0 <= 1"
    else map(x -> -x[1] <= x[2],
             map(dehomog, &?(NRays(P) >= 0, Hspaces(P), Hspaces(P)[1..-2])))
  fi
end;

POLYHEDRON[dual] := proc(P)
  if nargs > 1
    then ERRORnargs
    else CONE['`convert/POLYHEDRON`'](Dual(P))
  fi
end;

# conversion to polyhedron

POLYHEDRON[`convert/POLYHEDRON`] := proc(P)
  if nargs > 1
    then ERRORnargs
    else OBJ('POLYHEDRON', op(2..POLYHEDRON_NOPS, P))
  fi
end;

# basic methods

POLYHEDRON[ambientdim] := proc(P)
  if nargs > 1
    then ERRORnargs
    else Ambientdim(P)-1
  fi
end;

# containment

CONVEX[contains, `<=`] := proc(ineq, P::{POLYHEDRON, ray})
local ll, v;
  ll := affh2ray(ineq);
  if nargs > 2
    then ERRORnargs
  elif type(P, ray)
    then
      v := homog(P, 1);
      if nops(v) = nops(ll)
        then evalb(Dotprod(ll, v) >= 0)
        else ERRORnotsamespace
      fi
  elif Ambientdim(P) = nops(ll)
    then contains_ray(Dual(P), ll)
    else ERRORnotsamespace
  fi
end;

CONVEX[contains, `=`] := proc(eq, P::{POLYHEDRON, ray})
local ll, v;
  ll := affh2ray(eq);
  if nargs > 2
    then ERRORnargs
  elif type(P, ray)
    then
      v := homog(P, 1);
      if nops(v) = nops(ll)
        then evalb(Dotprod(ll, v) = 0)
        else ERRORnotsamespace
      fi
  elif Ambientdim(P) = nops(ll)
    then contains_line(Dual(P), ll)
    else ERRORnotsamespace
  fi
end;

POLYHEDRON[contains] := proc(P, arg2::{ray, affray, line, POLYHEDRON})
local d, r;
  d := Ambientdim(P);
  if nargs > 2
    then ERRORnargs
  elif type(arg2, ray)
    then
      r := homog(arg2, 1);
      if d = nops(r)
        then contains_ray(P, r)
        else ERRORnotsamespace
      fi
  elif type(arg2, affray)
    then
      r := homog(op(1, arg2), 0);
      if d = nops(r)
        then contains_ray(P, r)
        else ERRORnotsamespace
      fi
  elif type(arg2, line)
    then
      r := homog(op(1, arg2), 0);
      if d = nops(r)
        then contains_line(P, r)
        else ERRORnotsamespace
      fi
    else POLYHEDRON[`&<=`](arg2, P)
  fi
end;

POLYHEDRON[relint] := proc(P::POLYHEDRON1)
  if nargs > 1
    then ERRORnargs
    else affine(Relint(P)) # the point is always affine!
  fi
end;

#
# intersection
#

POLYHEDRON[_intersection] := proc(n, Cl, eqs, ineqs)
  NEWPOLYHEDRON(Intersection(n, Cl, eqs, [op(ineqs), origin(n)]))
end;

#
# convexhull
#

POLYHEDRON[_hull] := proc()
  NEWPOLYHEDRON(Hull(args));
end;

convexhull := proc()
# args: rays (= points) and/or affrays (= rays) and/or lines
#       and/or polyhedra and/or cones in any order
local al, Pl, r, l, n, L, C;
  al := [ args ];
  l := map(homog, map2(op, 1, select(type, al, line)), 0);
  r := [op(map(homog, map2(op, 1, select(type, al, affray)), 0)), # rays
        op(map(homog, select(type, al, ray), 1))];                # vertices
  Pl := [op(map(affinecone, select(type, al, CONE))),
         op(select(type, al, POLYHEDRON))];

  n := nops(l)+nops(r)+nops(Pl);
  if n = 0 or n <> nargs
    then ERRORraylinecone
  fi;

  L := [ op(l), op(r) ];
  n := &?(L <> [], nops(L[1]), Ambientdim(Pl[1]));
  if andmap(proc(x) Ambientdim(x) = n end, Pl) and
     (L = [] or (nops(L[1]) = n and type(L, listlist)))
    then
      C := Hull(n, Pl, l, r);
      r := Rays(C);
      if ormap(Isaffine, r) or (n <> 1 and Dim(C) = 0)
        # if n = 0, then Dim(C) = 0 is not sufficient
        then NEWPOLYHEDRON(C)
        else WARNINGnotaffine; emptypolyhedron(n-1)
      fi
    else ERRORnotsamespace
  fi
end;

#
# affine hull
#

CONVEX[affinehull] := proc()
local al, Cl, ll;
  al := [ args ];
  Cl := map(POLYHEDRON['`convert/CONE`'], select(type, al, POLYHEDRON));
  ll := [ op(map(homog, select(type, al, ray), 1)),
          op(map(homog, map(op, select(type, al, line)), 0)),
          op(map(homog, map(op, select(type, al, affray)), 0)) ];
  if nargs = 0 or nops(Cl)+nops(ll) <> nargs
    then ERRORarlp
    else CONE['`convert/POLYHEDRON`'](CONE['linearhull'](op(Cl), op(ll)))
      # not optimal!
  fi
end;

#
# image/preimage
#

`polyhedron/image/newvertex` := proc(r, c, v)
local ar, ir;
  ar, ir := op(dehomog(r));
  reduceqvec([op(c*ar+ir*v), ir])
end;

`polyhedron/image/newhspace` := proc(h, b, v)
local ah, ih;
  ah, ih := op(dehomog(h));
  ah := b*ah;
  reduceqvec([op(ah), ih-Dotprod(v, ah)])
end;

`polyhedron/image/similar` := proc(P::POLYHEDRON, c::rational, v::list)
# returns a true polyhedron
#   c::rational and not identical(0) # does not work!
local r, nr;
  r := Rays(P);
  nr := abs(NRays(P));
  OBJ('POLYHEDRON', Ambientdim(P), Lines(P),
    gausselim2(Lines(P),
               [&?(c > 0, op(r[1..nr]), op(-r[1..nr])),
                op(map(`polyhedron/image/newvertex`, r[nr+1..-1], c, v))]),
    Incidentfacets(P),
    gausselim(map(`polyhedron/image/newhspace`, Hplanes(P), 1/c, v),
              map(`polyhedron/image/newhspace`, Hspaces(P), 1/c, v)),
    Incidentrays(P),
    NRays(P))
end;

POLYHEDRON[image] := proc(P, A::{mat, ray, rational}, v::ray)
# returns a true polyhedron
local d, All, s, vl;
  d := POLYHEDRON['ambientdim'](P);
  if nargs > 3
    then ERRORnargs
  elif type(A, mat)
    then
      if A <> [] and (type(A, list) or mat_nrows(A) <> 0)
      # A <> [] and not (type(A, array) and op(matrix_OP_nrows, eval(A)) = 0)
        then
          All := convert(A, listlist);
#         if nargs = 2 then vl := [0$nops(All)] else vl := convert(v, list) fi;
          vl := `if`(nargs = 2, [0$nops(All)], convert(v, list));
          if nops(All[1]) <> d or nops(vl) <> nops(All)
            then ERRORwrongmap
            else NEWPOLYHEDRON(Image(P, mat_homog(All, vl)))
          fi
      elif (nargs = 3 and convert(v, list) <> []) or
           (not type(A, list) and op(matrix_OP_ncols, eval(A)) <> d)
        then ERRORwrongmap
      elif POLYHEDRON['isempty'](P)
        then emptypolyhedron(0)
        else fullpolyhedron(0)
      fi
    else # similarity transformation
      if type(A, rational)
        then
          s := A;
#          if nargs = 2 then vl := [0$d] else vl := convert(v, list) fi
          vl := `if`(nargs = 2, [0$d], convert(v, list))
      elif nargs = 2
        then
          s := 1;
          vl := convert(A, list)
        else
          ERRORillegalcomb
      fi;
      if nops(vl) <> d
        then ERRORwrongmap
      elif POLYHEDRON['isempty'](P)
        then P
      elif s <> 0
        then `polyhedron/image/similar`(P, s, vl)
        else NEWPOLYHEDRON(Hull(d+1, [], [], [homog(vl, 1)]))
     fi
  fi
end;

POLYHEDRON[preimage] := proc(P, A::{mat, ray, rational}, v::ray)
# returns a true polyhedron
local d, All, s, vl;
  d := POLYHEDRON['ambientdim'](P);
  if nargs > 3
    then ERRORnargs
  elif type(A, mat)
    then
      if A <> [] and (type(A, list) or mat_nrows(A) <> 0)
      # A <> [] and not (type(A, array) and op(matrix_OP_nrows, eval(A)) = 0)
        then
          All := convert(A, listlist);
#          if nargs = 2 then vl := [0$d] else vl := convert(v, list) fi;
          vl := `if`(nargs = 2, [0$d], convert(v, list));
          if nops(All) <> d or nops(vl) <> nops(All)
            then ERRORwrongmap
            else NEWPOLYHEDRON(Preimage(P, mat_homog(All, vl)))
          fi
      elif d <> 0 or (nargs = 3 and convert(v, list) <> [])
        then ERRORwrongmap
      elif A = []
        then ERRORambientdim0
      elif POLYHEDRON['isempty'](P)
	then emptypolyhedron(mat_ncols(A))
	else fullpolyhedron(mat_ncols(A))
      fi
    else # similarity transformation
      if type(A, rational)
        then
          s := A;
#          if nargs = 2 then vl := [0$d] else vl := convert(v, list) fi
          vl := `if`(nargs = 2, [0$d], convert(v, list))
      elif nargs = 2
        then
          s := 1;
          vl := convert(A, list)
        else
          ERRORillegalcomb
      fi;
      if nops(vl) <> d
        then ERRORwrongmap
      elif s <> 0
        then `polyhedron/image/similar`(P, 1/s, -vl/s)
      elif contains_ray(P, homog(vl, 1))
        then fullpolyhedron(d)
        else emptypolyhedron(d)
      fi
  fi
end;

#
# recession cone
#

POLYHEDRON[recession] := proc(P)
  if nargs > 1
    then ERRORnargs
    else reduce(NEWCONE(POLYHEDRON['ambientdim'](P),
                        POLYHEDRON['lines'](P), # automatically gausselim'ed
                        POLYHEDRON['rays'](P),
                        Incidentfacets(P)[1..abs(NRays(P))],
                        map(x -> dehomog(x)[1], Hplanes(P)),
                          # gausselim in reduce
                        map(x -> reducevec(dehomog(x)[1]), Hspaces(P)),
                        {}))
  fi
end;

#
# cartesian product and join
#

`polyhedron/&x/ray` := proc(v, ro)
  subsop(op(ro), v)
end;

`polyhedron/&x/hspace` := proc(h, ro, ho)
local i;
  subsop(op(ro), -1 = `+`(seq(h[i], i = ho))+h[-1], h)
end;

`polyhedron/&x/cone` := proc(P1)
# computes the product, but of type cone
local P, Pd, d, i, eqs, v, n, ro, ho;
  P := CONE[`&x`](op(map(CONE[`convert/CONE`], [args])));
  Pd := Ambientdim(P);
  eqs := NULL;
  n := Ambientdim(P1);
  v := -origin(n);
  ho := NULL;
  for i from 2 to nargs do
    ho := ho, n;
    d := Ambientdim(args[i]);
    eqs := eqs, embed(Pd, v, 0, origin(d), n);
    n := n+d;
  od;

  P := motzkin(P, [eqs], []);
  ho := [ho];
  ro := map(x -> x = NULL, ho);
  reduce(NEWCONE(Pd-nargs+1,
    map(`polyhedron/&x/ray`, Lines(P), ro),
      # CORRECT???
    map(`polyhedron/&x/ray`, Rays(P), ro),
    Incidentfacets(P),
    map(`polyhedron/&x/hspace`, Hplanes(P)[1..-nargs], ro, ho),
      # for &x as well as for join, no additional eq has been redundant
      # gausselim in reduce
    map(`polyhedron/&x/hspace`, Hspaces(P), ro, ho),
    Incidentrays(P)))
end;

POLYHEDRON[`&x`] := proc()
# args: P1::POLYHEDRON, ...
# not optimal: incidence structures could be computed directly
local Pl;
  Pl := [args];
  if nargs = 0
    then fullpolyhedron(0)
  elif not type(Pl, list(POLYHEDRON))
    then ERRORpoly
  elif ormap(POLYHEDRON['isempty'], Pl)
    then emptypolyhedron(`+`(op(map(CONE['ambientdim'], Pl)))-nargs)
    else NEWPOLYHEDRON(`polyhedron/&x/cone`(op(Pl)))
  fi
end;

POLYHEDRON[join] := proc()
# args: P1::POLYHEDRON, ...
local Pl;
  Pl := [args];
  if nargs = 0
    then fullpolyhedron(0)
  elif not type(Pl, list(POLYHEDRON))
    then ERRORpoly
    else NEWPOLYHEDRON(Dual(`polyhedron/&x/cone`(op(map(Dual, Pl)))))
  fi
end;

#
# tests for containment and similar routines
#

POLYHEDRON[containsrelint] := proc(P, r::ray)
  if nargs > 2
    then ERRORnargs
    else CONE['containsrelint'](P, homog(r, 1))
  fi
end;

#
# testing polyhedra
#

POLYHEDRON[isempty] := proc(P)
  if nargs > 1
    then ERRORnargs
    else evalb(Rays(P) = [])
  fi
end;

Isbounded := proc(P)
  if nargs > 1
    then ERRORnargs
    else NRays(P) = 0 and Lines(P) = []
  fi
end;

POLYHEDRON[isaffine] := proc(P)
  if nargs > 1
    then ERRORnargs
    else evalb(nops(Rays(P)) = 1)
  fi
end;

POLYHEDRON[islinear] := proc(P)
  if nargs > 1
    then ERRORnargs
    else evalb(Rays(P) = [origin(Ambientdim(P))])
      # due to the normalisation of rays, the "vertex" of a linear subspace
      # is always the origin
  fi
end;

POLYHEDRON[issimplex] := proc(P)
  if nargs > 1
    then ERRORnargs
    else NRays(P) = 0 and POLYHEDRON['issimplicial'](P)
  fi
end;

#
# faces
#

POLYHEDRON[facets] := proc(P)
local sat2, i;
  if nargs > 1 then ERRORnargs fi;
  sat2 := Incidentrays(P);
  [ seq(PFACE[NEW](sat2[i], {i}, P, 1),
        i = 1..nops(sat2)+&?(NRays(P) >= 0, 0, -1)) ]
end;

#
# volume
#

POLYHEDRON[volume] := proc(P)
  if nargs > 1
    then ERRORnargs
  elif Hplanes(P) = []
    then PFACE['volume'](POLYHEDRON['maximal'](P))
    else 0
  fi
end;

POLYHEDRON[surface] := proc(P)
local d;
  d := POLYHEDRON['codim'](P);
  if nargs > 1
    then ERRORnargs
  elif d = 0
    then PFACE['surface'](POLYHEDRON['maximal'](P))
  elif d = 1
    then 2*PFACE['volume'](POLYHEDRON['maximal'](P))
    else 0
  fi
end;

POLYHEDRON[distance] := proc(P, v::ray)
local vl, d, eq, i, h, h0, h2;
  vl := homog(v, 1);
  d := Ambientdim(P);
  eq := Hplanes(P);

  if nops(vl) <> d
    then ERRORnotsamespace
  elif nops(eq) = d # empty polyhedron
    then RETURN(infinity)
  fi;

  for i from nops(eq) to 1 by -1 do
    h := eq[i];
    h0 := subsop(X0 = 0, h);
    h2 := Dotprod(h0, h0);
    vl := reducevec(h2*vl-Dotprod(vl, h)*h0);
    eq := map(proc(x) reducevec(h2*x-Dotprod(subsop(X0 = 0, x), h0)*h) end,
              eq[1..i-1])
#    eq := map((x, y, y0, z) -> reducevec(z*x-Dotprod(subsop(X0 = 0, x), y0)*y),
#              eq[1..i-1], h, h0, h2)
  od;
  affine(vl);
# still work to do!!!
end;

#
# minkowski sum
#

POLYHEDRON[minkowskisum] := proc(P1)
# args: P1, ..., Pn::POLYHEDRON
# not optimal!
local Pl, d, P, vl, rl, v;
  if nargs = 1 then RETURN(POLYHEDRON['`convert/POLYHEDRON`'](P1)) fi;
  Pl := [args];
  d := POLYHEDRON['ambientdim'](P1);
  if not type(Pl, list(POLYHEDRON))
    then ERRORpoly
  elif ormap(proc(x) POLYHEDRON['ambientdim'](x) <> d end, Pl)
    then ERRORnotsamespace
  fi;
  vl := [[0$d]];
  rl := NULL;
  for P in Pl do
    vl := [seq(op(map(`+`, vl, v)), v = POLYHEDRON['vertices'](P))];
    rl := rl, op(map(`ray`, POLYHEDRON['rays'](P)));
  od;
  if vl = []
    then emptypolyhedron(d)
    else convexhull(op(vl), rl)
  fi
end;

#
# graphics
#

`convex/addangle` := proc(v, v1, v12, v13)
local w;
  w := v-v1;
  [ evalf(arctan(Dotprod(w, v12), Dotprod(w, v13))), v ]
end;

`convex/orderplanepoints` := proc(S::list)
# S must be a list of points which lie in a common plane
# yields a list of the given points in clockwise or counterclockwise order
local v0, L;
  if nops(S) < 4 then RETURN(S) fi;

  v0 := (S[1]+S[2])/2;
  # we can't use a vertex since we don't know
  # where to insert it later into the sorted list
  L := map(`convex/addangle`, S, v0, S[2]-v0, S[3]-v0);
  # we don't have to compute an orthonormal basis; we get the same up to
  # linear transformation (with v0 = origin), and that's all we need
  map2(op, 2, sort(L, (x,y) -> evalb(x[1] < y[1]) ))
end;

POLYHEDRON[plotdata] := proc(P::POLYTOPE)
# creates a PLOT or PLOT3D data structure
# optional 2nd arg: plot option or list of plot options
# returns NULL for the empty polyhedron
local d, opts;
  if nargs > 2
    then ERRORnargs
  elif not member(POLYHEDRON['ambientdim'](P), [2, 3])
    then ERRORdim23
  fi;

  opts := &?(nargs = 2, op(convert(args[2], PLOToptions)), NULL);
  d := dim(P);
  if d = 3
    then
      # concatenation with @ does not work since it returns unevaluated
      POLYGONS(op(map(`convex/orderplanepoints`,
                      map(PFACE['vertices'], POLYHEDRON['facets'](P)))),
               opts)
  elif d = 2
    then POLYGONS(`convex/orderplanepoints`(Vertices(P)), opts)
  elif d = 1
    then CURVES(Vertices(P), opts)
  elif d = 0
    then POINTS(op(Vertices(P)), opts)
  fi
end;

POLYHEDRON[draw] := proc(P)
local d;
  d := THIS['ambientdim'](P);
  try
    (&?(d = 2, PLOT, PLOT3D))(THIS['plotdata'](args))
  catch: error
  end try
end;

#
# proximum & distance
#

macro(proxi_g = `polyhedron/proximum/g`);

proxi_g := (v, w, w0) -> v-Dotprod(v, w0)*w;

POLYHEDRON[proximum] := proc(P::POLYHEDRON1, v::vec, dist::name)
local eq, ineq, w, w0, i, j, p, f, fl, fl1, fl2;
  p := homog(v, 1); # this will finally be the proximum
  if nargs > 3
    then ERRORnargs
  elif nops(p) <> Ambientdim(P)
    then ERRORnotsamespace
  fi;

  eq := Hplanes(P);
  ineq := Hspaces(P);
  for i to nops(eq) do
    w := eq[i];
    w0 := subsop(X0 = 0, w);
    w0 := w0/Dotprod(w0, w0);
    for j from i+1 to nops(eq) do
      eq[j] := proxi_g(eq[j], w, w0)
    od;
    ineq := map(proxi_g, ineq, w, w0);
    p := proxi_g(p, w0, w);
  od;

  f := POLYHEDRON['maximal'](P);
  do
    fl := PFACE['pred'](f);
    fl1 := map((x, y, z) -> y[(Hspacenos(x) minus z)[1]],
               fl, ineq, Hspacenos(f));
    fl2 := map(proc(h)
               local h0, c;
                 c := Dotprod(h, p);
                 if c >= 0
                   then 0
                   else
                     h0 := subsop(X0 = NULL, h);
                     c*c/Dotprod(h0, h0)
                 fi
               end,
               fl1);
    i := findmax(x -> x, fl2);
    if fl2[i] = 0
    then
      p := affine(p);
      if nargs = 3
        then
          w := convert(v, list)-p;
          dist := sqrt(Dotprod(w, w))
      fi;
      RETURN(p)
    fi;
    f := fl[i];
    w := fl1[i];
    w0 := subsop(X0 = 0, w);
    w0 := w0/Dotprod(w0, w0);
    ineq := map(proxi_g, ineq, w, w0);
    p := proxi_g(p, w0, w)
  od
end;

POLYHEDRON[distance] := proc(P, v::vec, p::name)
local d;
  try
    if POLYHEDRON['isempty'](P)
      then d := infinity
    elif nargs = 3
      then p := POLYHEDRON['proximum'](P, v, d)
      else POLYHEDRON['proximum'](P, v, d)
    fi;
  catch "" : error lasterror
  end try;
  d
end;

POLYHEDRON[genhvector] := proc(P::POLYHEDRON1)
  if nargs <> 1 then ERRORnargs fi;
  FAN['genhvector'](normalfan(P))
end;

#
# copy from CONE to POLYHEDRON
#

COPYNEWENTRIES(CONE, POLYHEDRON);
