#
# functions for various (homology) coefficients
#

#
# MODQ (rational)
#

MODQ[_char] := 0;

MODQ[_linrank] := proc(A)
#  linrank(convert(A, listlist))
#    # only very small overhead if A is already of type listlist
  LinearAlgebra:-LA_Main:-Rank(convert(A, Matrix))
end;

MODQ[_homology] := proc(i0::integer, i1::integer, A::Array, d::procedure)
# computes the RATIONAL homology of a complex between i0 and i1
# A[i] is the rank of the i-th degree
# d returns the differential A[i] -> A[i-1] as {listlist, array, Array}
# d(i) is not called if it must necessarily be the zero map 
local i, rk;
  for i from i0+1 to i1 do
    if A[i-1] = 0 or A[i] = 0 then next fi;
    rk := THIS[_linrank](d(i));
    A[i-1] := A[i-1]-rk;
    A[i]   := A[i]  -rk
  od;
  A
end;

#
# prime fields
#

CONVEX[_mod_p] := proc(p)
# returns a module with functions "_linrank" and "_homology"
# for linear algebra over the field Z/p.
option remember;
  module()
  option package;
  export _char, _linrank, _homology;

_char := p;

_linrank := proc(A)
local B, rk;
  B := LinearAlgebra:-Modular:-Mod(p, A, integer[]);
  LinearAlgebra:-Modular:-RowReduce(p, B, Matrix_nrows(B), Matrix_ncols(B),
                                    0, 0, 0, rk, 0, 0, false);
$ifdef MINT
  rk := 0;
$endif
  rk
end;

_homology := proc(i0::integer, i1::integer, A::Array, d::procedure)
# computes the homology (with coeffs in a field) of a complex between i0 and i1
# A[i] is the rank of the i-th degree
# d returns the differential A[i] -> A[i-1] as {listlist, array, Array}
# d(i) is not called if it must necessarily be the zero map 
local i, rk;
  for i from i0+1 to i1 do
    if A[i-1] = 0 or A[i] = 0 then next fi;
    rk := _linrank(d(i));
    A[i-1] := A[i-1]-rk;
    A[i]   := A[i]  -rk
  od;
  A
end;

  end module
end;

#
# coefficient parsing
#         

parsemod := proc()
# if the last arg is "integer", returns nargs-1, MODZ
# if the last arg is "integer(p)", returns nargs-1, CONVEX['_mod_p'](p)
# if the last arg is "rational", returns nargs-1, MODQ
# otherwise, returns nargs, MODQ
# at least one args must be given
local c;
  c := args[-1];
  if c = 'integer'
    then nargs-1, MODZ
  elif c = 'rational'
    then nargs-1, MODQ
  elif type(c, specfunc(prime, integer)) and nops(c) = 1
    then
      nargs-1, CONVEX['_mod_p'](op(1, c))
    else nargs, MODQ
  fi
end;
