#
# CONES
#

#
# type CONE
#

`type/CONE` := proc(x)
  op(0, x) = OBJ and op(1, x)[BASE] = 'CONE'
end;

`type/CONE1` := proc(x)
  `type/CONE`(x) and Rays(x) <> []
end;

`type/CONE0` := proc(x)
  `type/CONE`(x) and Lines(x) = []
end;

#macro(NEWCONE = `convex/NEWCONE`);
#
#NEWCONE := proc(d::posint, l::list, r::list, sat::list,
#  eq::list, ineq::list, sat2::list)
#  'cone'(args)
#end;

# macro(NEWCONE = 'cone');

NEWCONE := proc()
  OBJ('CONE', args)
end;

CONE[_print] := proc(d::nonnegint, l::list, r::list, sat::list,
                       eq::list, ineq::list, sat2::list)
$ifdef MINT
  sat = sat2;
$endif
  THIS(d, d-nops(eq), nops(l), nops(r), nops(ineq))
end;

CONE[BASE] := 'CONE';
CONE[SHIFT] := 0;
CONE[EL] := 'CFACE';
CONE[POSDOM] := 'FAN'; # possible domain

# basic cones

zerocone := proc(d::nonnegint)
options remember;
  if nargs > 1
    then ERRORnargs
    else NEWCONE(d, [], [], [], basiclists(d), [], [])
  fi
end;

fullcone := proc(d::nonnegint)
options remember;
  if nargs > 1
    then ERRORnargs
    else NEWCONE(d, basiclists(d), [], [], [], [], [])
  fi
end;

# basic operations

CONE[lines] := proc(C)
  if nargs > 1
    then ERRORnargs
    else Lines(C)
  fi
end;

CONE[rays] := proc(C)
  if nargs > 1
    then ERRORnargs
    else Rays(C)
  fi
end;

CONE[incidentfacets] := proc(C)
  if nargs > 1
    then ERRORnargs
    else Incidentfacets(C)
  fi
end;

CONE[hplanes] := proc(C)
  if nargs > 1
    then ERRORnargs
    else Hplanes(C)
  fi
end;

CONE[hspaces] := proc(C)
  if nargs > 1
    then ERRORnargs
    else Hspaces(C)
  fi
end;

CONE[incidentrays] := proc(C)
# used in 'polyhedron/image/similar' as "op"
  if nargs > 1
    then ERRORnargs
    else Incidentrays(C)
  fi
end;

# dualising and converting to true cone type

CONE[dual] := proc(C)
  if nargs > 1
    then ERRORnargs
    else NEWCONE(op(CONE_OP_ambientdim, C),
          	 op(CONE_OP_hplanes, C),
          	 op(CONE_OP_hspaces, C),
          	 op(CONE_OP_incidentrays, C),
          	 op(CONE_OP_lines, C),
          	 op(CONE_OP_rays, C),
          	 op(CONE_OP_incidentfacets, C))
  fi
end;

# conversion to polyhedron

CONE[`convert/POLYHEDRON`] := proc(C)
local d;
  d := Ambientdim(C);
  if nargs > 1
    then ERRORnargs
  elif d = 0
    then ERRORdim0
    else POLYHEDRON['_intersection'](d, [C], [], [])
  fi
end;

# conversion to affine cone

affinecone := proc(C, v::ray)
local P, vl;
  P := OBJ('POLYHEDRON',
           op(2..-1, CONE[`&x`](C, posorthant(1))), -nops(Rays(C)));
  if nargs > 2
    then ERRORnargs
  elif nargs = 1
    then return P
  fi;
  vl := convert(v, list);
  if nops(vl) = Ambientdim(C)
    then `polyhedron/image/similar`(P, 1, vl)
    else ERRORnotsamespace
  fi
end;

# subtypes

# `type/pointedcone` := proc(x)
#   type(x, CONE) and Lines(x) = []
# end;

# `type/fulldimcone` := proc(x)
#   type(x, CONE) and Hplanes(x) = []
# end;

#
# positive hull
#

hull := proc()
# args: rays and/or lines and/or cones and/or polyhedra in any order
local al, Cl, Pl, r, l, n, L;
  al := [ args ];
  l := map(op, select(type, al, line));
  r := [op(map(reduceqvec, select(type, al, ray))),
	op(map(op, select(type, al, affray)))];
  Cl := select(type, al, CONE);
  Pl := select(type, al, POLYHEDRON);

  n := nops(l)+nops(r)+nops(Cl)+nops(Pl);
  if n = 0 or n <> nargs
    then ERRORraylinecone
  fi;

  L := [ op(l), op(r) ];
  n := &?(L <> [], nops(L[1]),
          &?(Cl <> [], Ambientdim(Cl[1]), Ambientdim(Pl[1])-1));
  if andmap(proc(x) Ambientdim(x) = n end, Cl) and
     andmap(proc(x) Ambientdim(x) = n+1 end, Pl) and
     (L = [] or (nops(L[1]) = n and type(L, listlist)))
    then Hull(n, Cl, [op(map(x -> op(POLYHEDRON['lines'](x)), Pl)), op(l)],
           [op(map(x -> op(map(y -> reducevec(dehomog(y)[1]), Rays(x))), Pl)),
            op(r)])
    else ERRORnotsamespace
  fi
end;

#
# linear hull
#

CONVEX[linearhull] := proc()
local al, Cl, ll, d, C;
  al := [ args ];
  Cl := select(type, al, CONE);
  ll := map(reduceqvec, [ op(select(type, al, ray)),
                          op(map(op, select(type, al, affray))),
                          op(map(op, select(type, al, line))) ]);
  if nargs = 0 or nops(Cl)+nops(ll) <> nargs
    then ERRORraylinecone
  fi;

  d := &?(ll = [], Ambientdim(Cl[1]), nops(ll[1]));
  if ormap(proc(x) Ambientdim(x) <> d end, Cl) or
     (ll <> [] and (nops(ll[1]) <> d or not type(ll, listlist)))
    then
      ERRORnotsamespace
    else
      Hull(d,
           [],
           [seq(op(Lines(C)), C = Cl), seq(op(Rays(C)), C = Cl), op(ll)],
           [])
  fi
end;

CONE[lineality] := proc(C)
# not optimal!
  if nargs > 1
    then ERRORnargs
    else CFACE[DOMTYPE](CONE['minimal'](C))
  fi
end;

# pointedcone := proc(C::cone)
# # returns the same type as C
#   subsop(CONE_OP_lines = [],
#          CONE_OP_hplanes = [op(Hplanes(C)), op(Lines(C))],
#          C)
# end;

#
# functions for (pre)images
#

Image := proc(C, A::{mat, rational, real_infinity})
local newr, newl, Ainvt, All;
  if nargs > 2
    then ERRORnargs
  elif type(A, rational)
    then
      if A = 0
        then zerocone(Ambientdim(C))
      elif A > 0
        then CONE[`convert/CONE`](C)
      elif A < 0
        then
          NEWCONE(Ambientdim(C),
	          Lines(C), -Rays(C), Incidentfacets(C),
	          Hplanes(C), -Hspaces(C), Incidentrays(C))
      fi
  elif type(A, real_infinity)
    then fullcone(Ambientdim(C))
  elif type(A, {matrix, Matrix}) and mat_nrows(A) = 0
    then
      if mat_ncols(A) <> Ambientdim(C)
        then ERRORwrongmap
        else fullcone(0)
      fi
  elif A = []
    then
      fullcone(0)
    else
      All := &?(type(A, list), A, convert(A, listlist));
      if nops(All[1]) <> Ambientdim(C) then ERRORwrongmap fi;
      newl := matmatmul(All, Lines(C));
      newr := map(reduceqvec, matmatmul(All, Rays(C)));
      Ainvt := inverse(A);
      if Ainvt <> FAIL
        then # A is invertible
          userinfo(2, CONE, '`regular map`');
          Ainvt := transpose(Ainvt);
          NEWCONE(nops(All[1]),
                  gausselim(newl, newr),
                  Incidentfacets(C),
                  gausselim(matmatmul(Ainvt, Hplanes(C)),
                            matmatmul(Ainvt, Hspaces(C))),
                  Incidentrays(C))
        else # A is not invertible
          userinfo(2, CONE, '`singular map`');
          Hull(nops(All), [], map(reduceqvec, newl), newr)
      fi
  fi
end;

Preimage := proc(C, A::{mat, rational, real_infinity})
local neweq, newineq, Ainv, At;
  if nargs > 2
    then ERRORnargs
  elif type(A, {rational, real_infinity})
    then Image(C, &?(A = 0, infinity, 1/A))
  elif A = []
    then
      if Ambientdim(C) = 0 then ERRORambientdim0 else ERRORwrongmap fi
  elif type(A, {matrix, Matrix}) and mat_nrows(A) = 0
    then
      if Ambientdim(C) = 0
        then fullcone(mat_ncols(A))
        else ERRORwrongmap
      fi
    else
      if mat_nrows(A) <> Ambientdim(C) then ERRORwrongmap fi;
      At := transpose(A);
      neweq := matmatmul(At, Hplanes(C));
      newineq := map(reduceqvec, matmatmul(At, Hspaces(C)));
      Ainv := inverse(A);
      if Ainv <> FAIL
        then # A is invertible
          userinfo(2, CONE, '`regular map`');
          NEWCONE(nops(At[1]),
                  gausselim(matmatmul(Ainv, Lines(C)),
                            matmatmul(Ainv, Rays(C))),
                  Incidentfacets(C),
                  gausselim(neweq, newineq),
                  Incidentrays(C))
        else # A is not invertible
          userinfo(2, CONE, '`singular map`');
          Intersection(nops(At), [], map(reduceqvec, neweq), newineq)
      fi
  fi
end;

## NEW IMAGE
#
#`convex/gauss` := proc(A)
#local getrow, B, m, n, r, i, j, v, s, U, t, w, V;
#  m := nops(A);
#  if m = 0 or A[1] = [] then RETURN(A, 0) fi;
#
#  getrow := proc(j::evaln)
#  local i, l, s;
#    if r > n then RETURN(false) fi;
#    for i from r to m do
#      if B[i, r] <> 0 then j := i; RETURN(true) fi
#    od;
#    for l from n to r+1 by -1 do
#      for i from r to m do
#        if B[i, l] <> 0
#          then
#            j := i; n := l-1;
#            for i to m do s := B[i, r]; B[i, r] := B[i, l]; B[i, l] := s od;
#            V := subsop(r = V[l], l = V[r], V);
#            RETURN(true)
#        fi
#      od
#    od;
#    false
#  end;
#
#  B := A;
#  n := nops(B[1]);
#  U := basiclists(m);
#  V := basiclists(n);
#  for r while getrow(j) do
#    v := B[j]/B[j, r]; w := U[j]/B[j, r];
#    s := r = v; t := r = w;
#    for i to r-1 do
#      if B[i, r] <> 0
#        then s := s, i = B[i]-B[i, r]*v; t := t, i = U[i]-B[i, r]*w; fi
#    od;
#    for i from j+1 to m do
#      if B[i, r] <> 0
#        then s := s, i = B[i]-B[i, r]*v; t := t, i = U[i]-B[i, r]*w; fi
#    od;
#    if j <> r then s := s, j = B[r]; t := t, j = U[r] fi;
#    B := subsop(s, B); U := subsop(t, U);
#  od;
#  r := r-1;
#  V := [op(1..r, V), seq(V[j]-`+`(seq(B[i, j]*V[i], i = 1..r)),
#        j = r+1..nops(B[1]))];
#  r, U, V;
#end;
#
#convex[image2] := proc(C::CONE, A::listlist)
#local r, U, V, d, d2, C2, B, i, j, k, ii, A2;
#  r, U, V := `convex/gauss`(A);
#  A2 := ilcm(op(map(denom, map(op, A))))*A;
#  U := ilcm(op(map(denom, map(op, U))))*U;
#  V := ilcm(op(map(denom, map(op, V))))*V;
##lprint(nops(V)-r);
##lprint(V);
##t := time();
#  d2 := nops(A); d := Ambientdim(C);
#  if V = basiclists(d) # (we use remember tables!)
#    then
#      B := [seq([seq(U[j, i], j = 1..d)], i = 1..d2)]
#    else
#      B := [seq([seq(`+`(seq(U[k, i]*V[k, j], k = 1..r)), j = 1..d)],
#                i = 1..d2)];
#  fi;
##B_ := B;
##print(time()-t); t := time();
##lprint(V[r+1..-1]);
#  C2 := motzkin(Dual(C), V[r+1..-1], []);
##print(time()-t); t := time();
#  if Incidentfacets(C2) <> Incidentrays(C)
#    then # combinatorial structure may have changed
#      C2 := reduce(C2)
#  fi;
#  C2 := Dual(C2);
##lprint("R", time()-t); t := time();
#
##lprint("M", time()-t); t := time();
#
#  C2 := normalise(NEWCONE(d2,
#          gausselim(matmatmul(A2, Lines(C2))),
#          map(reducevec, matmatmul(A2, Rays(C2))),
#          Incidentfacets(C2),
#          gausselim([op(matmatmul(B, Hplanes(C2))), op(r+1..-1, U)]),
#          map(reducevec, matmatmul(B, Hspaces(C2))),
#          Incidentrays(C2)
#         ));
##lprint("N", time()-t); t := time();
#  C2
#end;

#
# tests for containment and similar routines
#

contains_ray := proc(C, r)
  andmap(proc(x) Dotprod(x, r)  = 0 end, Hplanes(C)) and
  andmap(proc(x) Dotprod(x, r) >= 0 end, Hspaces(C))
end;

CONE[containsrelint] := proc(C, r::ray)
  if nargs > 2
    then ERRORnargs
    else andmap(proc(x) Dotprod(x, r) = 0 end, Hplanes(C)) and
         andmap(proc(x) Dotprod(x, r) > 0 end, Hspaces(C))
  fi
end;

contains_line := proc(C, r)
# r::ray
local l;
  l := Lines(C);
  Ambientdim(C) = 0 or linrank([op(l), r]) = nops(l)
end;

CONVEX[contains, list] := proc(r, C::CONE)
local rl;
  rl := reduceqvec(r);
  if nargs > 2
    then ERRORnargs
  elif Ambientdim(C) = nops(rl)
    then contains_ray(Dual(C), r)
    else ERRORnotsamespace
  fi
end;

CONVEX[contains, array] := eval(CONVEX[contains, list]);

CONVEX[contains, Vector[column]] := eval(CONVEX[contains, list]);

CONVEX[contains, Vector[row]] := eval(CONVEX[contains, list]);

CONVEX[contains, line] := proc(l, C::CONE)
local ll;
  ll := reduceqvec(op(1, l));
  if nargs > 2
    then ERRORnargs
  elif Ambientdim(C) = nops(ll)
    then contains_line(Dual(C), ll)
    else ERRORnotsamespace
  fi
end;

CONE[contains] := proc(C, arg2::{ray, line, CONE})
local d, r;
  d := Ambientdim(C);
  if nargs > 2
    then ERRORnargs
  elif type(arg2, ray)
    then
      r := reduceqvec(arg2);
      if d = nops(r)
        then contains_ray(C, r)
        else ERRORnotsamespace
      fi
  elif type(arg2, line)
    then
      r := op(arg2);
      if d = nops(r)
        then contains_line(C, r)
        else ERRORnotsamespace
      fi
    else CONE['`&<=`'](arg2, C)
  fi
end;

Relint := proc(C)
local r;
  r := Rays(C);
  if nargs > 1
    then ERRORnargs
  elif r = []
    then [0$ Ambientdim(C)]
    else reducevec(`+`(op(r)))
  fi
end;

#
# cartesian product
#

`cone/&x/addincidence` := proc(CS::set, CAS::set, AS::set, i::nonnegint)
  AS minus map(`+`, CAS minus CS, i)
end;

CONE[`&x`] := proc()
local C, Cl, d, l, r, sat, eq, ineq, sat2, nr, ni, i, ir, ii, Sr, Sineq;
  Cl := [ args ];

  if not type(Cl, list(CONE))
    then ERRORillegalcomb
  elif nargs = 1
    then return CONE[`convert/CONE`](Cl[1])
  fi;

  d := `+`(op(map(CONE['ambientdim'], Cl)));
  Sr    := { $1..`+`(seq(nops(Rays(C)), C = Cl)) };
  Sineq := { $1..`+`(seq(nops(Hspaces(C)), C = Cl)) };

  l := NULL; r := NULL; sat := NULL;
  eq := NULL; ineq := NULL; sat2 := NULL;
  i := 0; ir := 0; ii := 0;
  for C in Cl do
    l    := l,    op(map2(embed, d, Lines(C), i));
    eq   := eq,   op(map2(embed, d, Hplanes(C), i));
    r    := r,    op(map2(embed, d, Rays(C), i));
    ineq := ineq, op(map2(embed, d, Hspaces(C), i));

    nr := nops(Rays(C));
    ni := nops(Hspaces(C));

    sat  :=  sat, op(map(`cone/&x/addincidence`, Incidentfacets(C),
                         { $1..ni }, Sineq, ii));
    sat2 := sat2, op(map(`cone/&x/addincidence`, Incidentrays(C),
                         { $1..nr }, Sr, ir));

    i  := i +Ambientdim(C);
    ir := ir+nr;
    ii := ii+ni;
  od;

  NEWCONE(d, [ l ], [ r ], [ sat ], [ eq ], [ ineq ], [ sat2 ])
    # [l] and [eq] are automatically gausselim'ed
end;

#
# testing
#

CONE[islinear] := proc(C)
  if nargs > 1
    then ERRORnargs
    else evalb(Rays(C) = [])
  fi
end;

#iszerocone := proc(C::cone)
#  evalb(Dim(C) = 0)
#end;

#
# faces
#

CONE[facets] := proc(C)
local sat2, i;
  if nargs > 1 then ERRORnargs fi;
  sat2 := Incidentrays(C);
  [seq(CFACE[NEW](sat2[i], {i}, C, 1), i = 1..nops(sat2))]
end;

CONE[edges] := eval(CONE[_edges]);

#
# test of regularity
#

CONE[isregular] := proc(C, basis::name)
local d, r, U, i;
  r := THIS['rays'](C);
  d := THIS['dim'](C);
  if nargs > 2
    then ERRORnargs
  elif nops(r) = d and  THIS['ispointed'](C)
    then
      r := transpose(r, THIS['ambientdim'](C));
      U := &?(nargs = 2, hermite2inv(r, true, basis), hermite1(r, false));
      for i to d do
        if abs(U[i, i]) <> 1 then return false fi
	# the "abs" is needed if nargs = 1
	# because then we only compute an upper triangular form
      od;
      true
    else
      false
  fi
end;

#
# Hilbert basis
#

# this algorithm is very poor!

`convex/lindiosolve/<=` := proc(x::list, y::list)
local i;
  for i from 1 to nops(x) do
    if x[i] > y[i] then return false fi
  od;
  true
end;

`convex/lindiosolve` := proc(rd::posint, eq::list, ineq::list, RS::set)
# the optional argument RS is the set of rays of the cones defined by eq & ineq
# if RS is not given, the list of all minimal solutions is returned
# otherwise a list containing only the first solution not in RS
local d, A, At, i, M, P, y, Ay, yinc, v0;
  userinfo(3, CONE, nops(eq), '`equations and`', nops(ineq),
                      '`inequalities in`', rd, '`dimensions`');
  if eq = [] and ineq = []
    then return `&?`(nargs = 4, [], basiclists(rd))
  fi;

  # converting inequalities to equations by introducing slack variables
  d := nops(ineq);
  A := [op(eq), seq([op(-ineq[i]), 0$i-1, 1, 0$d-i], i = 1..d)];
  At := transpose(A);
  d := d+rd;

  # solving the linear diophantine systems using the
  # breadth-first version of the Contejean-Devie algorithm
  P := basiclists(d);
# for lattice points:
#  P := [basiclists(d)[rd]];
  M := [];
  v0 := [0$nops(A)];
  while P <> [] do
#lprint(P);
    y := P[1];
    P := subsop(1 = NULL, P);
    Ay := matvecmul(A, y);
    if Ay = v0
      then # solution
        if not member(y, M)
          then
            # optional argument RS ?
            if nargs = 4 and not member(y[1..rd], RS)
              then return [y[1..rd]]
            fi;
            M := [op(M), y]
        fi
    elif not ormap(proc(x) `convex/lindiosolve/<=`(x, y) end, M)
      then # develop
        for i to d do
          if Dotprod(At[i], Ay) < 0
# for lattice points:
#          if i <> rd and Dotprod(At[i], Ay) < 0
            then
              yinc := subsop(i = y[i]+1, y);
              if not member(yinc, P) then P := [ op(P), yinc ] fi
          fi
        od;
    fi
  od;
  map(proc(x) x[1..rd] end, M)
end;

`convex/hilbertbasis` := proc(all::boolean, C::CONE, B::name)
# B is optional and receives a lattice basis of the lineality space
local l, r, d, U, V, ineq, C2;
  d := Ambientdim(C);

  # find a transformation which maps the rays into the positive orthant
  # we only know how to do this for cones which are the sum of its
  # lineality space and a simplicial cone. Therefore, we first pass to
  # a larger cone which is simplicial.

  C2 := Intersection(d, [], Hplanes(C), selectbasis(Hspaces(C)));
  l := Lines(C2);
  r := Rays(C2);
  hermite3(transpose([op(l), op(r)], d), true, U, V);
  # U and V now are the transformation matrices we were looking for

  if nargs = 3 then B := V[1..nops(l)] fi;
  if r = [] then return [] fi;

  d := Dim(C);
  # ignore lineality space
  U := U[nops(l)+1..d];
  V := V[nops(l)+1..d];
  d := d-nops(l);

  # We don't need equations because they just say that the last new
  # coordinates must be 0. But we have already restricted ourselves
  # to Dim(C) instead of Ambientdim(C).
  ineq := remove(member, map(reducevec, matmatmul(V, Hspaces(C))),
                 convert(basiclists(d), set));
  matmatmul(transpose(V), `convex/lindiosolve`(d, [], ineq,
    &?(all, NULL, convert(matmatmul(U, r), set))))
end;

CONE[hilbertbasis] := proc(C, B::name)
# B is optional and receives a lattice basis of the lineality space
  if nargs > 2
    then ERRORnargs
    else `convex/hilbertbasis`(true, args)
  fi
end;

CONE[hvector] := eval(SWITCH2FAN);

CONE[genhvector] := eval(SWITCH2FAN);
