#
# definition of some examples
#

posorthant := proc(d::nonnegint)
option remember;
local r, S, sat, i;
  if nargs > 1 then ERRORnargs fi;
  r := basiclists(d);
  S := { $1..d };
  sat := [ seq(S minus {i}, i = 1..d) ];
  NEWCONE(d, [], r, sat, [], r, sat)
end;

CONVEX[cube] := proc(d::nonnegint, c::rational)
option remember;
local f, l, i, v;
  if nargs > 2
    then ERRORnargs
  elif d = 0
    then RETURN(fullpolyhedron(0))
  elif nargs = 1
    then v := 1
    else v := c
  fi;
  f := proc(l, c) [op(l), c] end;
  l := [[-v], [v]];
  for i from 2 to d do l := [ op(map(f, l, -v)),  op(map(f, l, v)) ] od;
$ifdef MINT
  i = 0;
$endif
  convexhull(op(l))
end;

CONVEX[crosspolytope] := proc(d::nonnegint, c::rational)
option remember;
local i, v;
  if nargs > 2
    then ERRORnargs
  elif d = 0
    then RETURN(fullpolyhedron(0))
  elif nargs = 1
    then v := 1
    else v := c
  fi;
  convexhull(seq([0$i-1, v, 0$d-i], i = 1..d),
             seq([0$i-1, -v, 0$d-i], i = 1..d))
end;

CONVEX[stdsimplex] := proc(d::posint)
# optional: entry::integer
option remember;
local i, v;
  if nargs > 2
    then ERRORnargs
  elif nargs = 1
    then v := 1
    else v := args[2]
  fi;
  convexhull([0$d], seq([0$i-1, v, 0$d-i], i = 1..d))
end;

CONVEX[cyclicpolytope] := proc(d::posint, n::posint)
local i, j;
  if nargs > 2
    then ERRORnargs
    else convexhull(seq([seq(i^j, j = 1..d)], i = 0..n-1))
  fi
end;

CONVEX[permutahedron] := proc(d::nonnegint)
  if nargs > 1
    then ERRORnargs
    else convexhull(op(map2(subsop, -1 = NULL,
                            combinat['permute']([$1..d+1]))))
  fi
end;

CONVEX[randompolytope] := proc(d::nonnegint, n::nonnegint, max::posint)
local c, i, j;
  if nargs > 3
    then ERRORnargs
  elif n = 0
    then emptypolyhedron(d)
  elif d = 0
    then fullpolyhedron(0)
    else
      c := rand(-max..max);
      convexhull(seq([seq(c(), j = 1..d)], i = 1..n))
  fi
end;

#
# read & write polymake format
#

`convex/writepoly/matrix` := proc(fd, sec, fmt, m)
local r, v;
#  if m = [] then RETURN() fi;
  fprintf(fd, "\n%s\n", sec);
  for r in m do
    v := dehomog(r);
    fprintf(fd, fmt, v[2], op(v[1]))
  od
end;

`convex/writepoly/set` := proc(fd, s)
local i;
  fprintf(fd, "{");
  for i in sort(convert(s, list)) do
    fprintf(fd, " %d", i-1)
  od;
  fprintf(fd, " }\n")
end;

CONVEX['writepoly'] := proc(P::POLYHEDRON1, fn::string, info::string)
local fd, fmt, d;
  if nargs > 3
    then ERRORnargs
  elif not Ispointed(P)
    then ERRORppointed
  fi;
  fd := fopen(cat(fn, ".poly"), WRITE);
  fprintf(fd, "# created by the Maple package `Convex' version " VERSION "\n");

  d := Ambientdim(P);
  fmt := cat(" %d" $ d, "\n");

  if nargs = 3
    then fprintf(fd, "\nDESCRIPTION\n%s\n", info)
  fi;

  fprintf(fd,
    "\nAMBIENT_DIM\n%d\n\nDIM\n%d\n\nN_VERTICES\n%d\n\nN_FACETS\n%d\n",
    d-1, POLYHEDRON['dim'](P), nops(Rays(P)), nops(Hspaces(P)));

  `convex/writepoly/matrix`(fd, "VERTICES", fmt, Rays(P));

  `convex/writepoly/matrix`(fd, "FACETS", fmt, Hspaces(P));

  `convex/writepoly/matrix`(fd, "AFFINE_HULL", fmt, Hplanes(P));

  fprintf(fd, "\nVERTICES_IN_FACETS\n");
  map2(`convex/writepoly/set`, fd, Incidentrays(P));

  fprintf(fd, "\nFAR_FACE\n");
  `convex/writepoly/set`(fd, {$1..abs(NRays(P))});

  fclose(fd);
end;

`convex/readpoly/readline` := proc(fd)
local l;
  do
    l := readline(fd);
    if l = 0
      then RETURN("")
    elif l = "" or l[1] <> "#"
      then RETURN(l)
    fi
  od
end;

`convex/readpoly/matrix` := proc(fd)
local l, i, fmt, v, vs;
  l := `convex/readpoly/readline`(fd);
  i := 1;
  fmt := "";
  do
    v := sscanf(l[i..-1], "%a%n");
    if v = 0 then break fi;
    fmt := cat(fmt, "%a");
    i := i+v[2];
  od;
  vs := NULL;
  do
    v := sscanf(l, fmt);
    if v = 0
      then break
    elif not type(v, list(rational))
      then fclose(fd); ERRORprational
    fi;
    vs := vs, homog(v[2..-1], v[1]);
    l := `convex/readpoly/readline`(fd)
  od;
  [vs]
end;

`convex/readpoly/listofset` := proc(fd)
local l, n, ss, se, c;
  ss := NULL;
  do
    l := `convex/readpoly/readline`(fd);
    if l = "" then break fi;
    n := sscanf(l, "{%n")[1]+1;
    se := NULL;
    do # we read up to 10 elements at once
      c := sscanf(l[n..-1], "%d%d%d%d%d%d%d%d%d%d%n");
      if nops(c) <> 11 then se := se, op(c); break fi;
      se := se, op(1..-2, c);
      n := n+c[-1]
    od;
    ss := ss, convert(map(`+`, [se], 1), set);
  od;
  [ss]
end;

`convex/readpoly/nops` := proc(sec, data, n::evaln)
  if assigned(data[sec]) and data[sec] <> [] then n := nops(data[sec][1]) fi
end;

`convex/readpoly/makesat` := proc(rl, ie)
local s, r;
  userinfo(3, POLYHEDRON, '`computing incidence information`');

  s := {$1..nops(ie)};
  [seq(select(proc(x) Dotprod(r, ie[x]) = 0 end, s), r = rl)]
end;

`convex/readpoly/matrixsecs` :=
  {'POINTS', 'VERTICES', 'INEQUALITIES', 'FACETS', 'AFFINE_HULL', 'EQUATIONS'};

CONVEX['readpoly'] := proc(fn::string)
local fd, sec, l, data, n, d, P, sat, sat2;
  if nargs > 1 then ERRORnargs fi;
  fd := fopen(cat(fn, ".poly"), READ);
  while not feof(fd) do
    sec := sscanf(`convex/readpoly/readline`(fd), "%s");
    if sec = 0 then next fi;
    sec := convert(sec[1], name);
    if member(sec, `convex/readpoly/matrixsecs`)
      then
        userinfo(4, POLYHEDRON, '`reading section`', sec);
        data[sec] := `convex/readpoly/matrix`(fd)
    elif member(sec, {'AMBIENT_DIM', 'DIM',    # 'IN_EQ_DIM',
                      'N_VERTICES', 'N_FACETS'})
      then
        userinfo(4, POLYHEDRON, '`reading section`', sec);
        data[sec] := sscanf(`convex/readpoly/readline`(fd), "%d")[1]
    elif member(sec, {'VERTICES_IN_FACETS', 'FACETS_THRU_VERTICES'})
      then
        userinfo(4, POLYHEDRON, '`reading section`', sec);
        data[sec] := `convex/readpoly/listofset`(fd)
    elif sec = 'DESCRIPTION' and
	assigned(infolevel[POLYHEDRON]) and infolevel[POLYHEDRON] >= 2
      then
        userinfo(2, POLYHEDRON, '`displaying section`', sec);
        do
          l := `convex/readpoly/readline`(fd);
          if l = "" then break fi;
          lprint(convert(l, name))
        od
      else
        userinfo(4, POLYHEDRON, '`skipping section`', sec);
        while sscanf(`convex/readpoly/readline`(fd), "%s") <> 0 do od
    fi;
  od;
  fclose(fd);

  if assigned(data['AMBIENT_DIM'])
    then
      n := data['AMBIENT_DIM']+1
    else
      map(`convex/readpoly/nops`, `convex/readpoly/matrixsecs`, data, n)
  fi;

  if not assigned(n) then ERRORconstruct fi;

  if assigned(data['AFFINE_HULL'])
    then data['AFFINE_HULL'] := gausselim(data['AFFINE_HULL'])
    # we need it gausselim'ed!
  elif assigned(data['EQUATIONS'])
    then data['AFFINE_HULL'] := gausselim(data['EQUATIONS'])
  fi;

  if assigned(data['DIM'])
    then d := data['DIM']+1
  elif assigned(data['AFFINE_HULL'])
    then d := nops(data['AFFINE_HULL'])
  elif assigned(data['VERTICES'])
    then d := linrank(data['VERTICES'])
  fi;

  if assigned(d) and n = d
    then data['AFFINE_HULL'] := []
  fi;

  if assigned(data['VERTICES']) and assigned(data['FACETS'])
     and assigned(data['AFFINE_HULL'])
    then
      if assigned(data['VERTICES_IN_FACETS'])
        then
          sat2 := data['VERTICES_IN_FACETS'];
          sat := `&?`(assigned(data['FACETS_THRU_VERTICES']),
	    data['FACETS_THRU_VERTICES'],
            transposesat(sat2, nops(data['VERTICES'])))
      elif assigned(data['FACETS_THRU_VERTICES'])
        then
          sat := data['FACETS_THRU_VERTICES'];
          sat2 := transposesat(sat, nops(data['FACETS']))
      elif nops(data['VERTICES']) < nops(data['FACETS'])
        then
          sat2 := `convex/readpoly/makesat`(data['FACETS'], data['VERTICES']);
          sat := transposesat(sat2, nops(data['VERTICES']))
        else
          sat := `convex/readpoly/makesat`(data['VERTICES'], data['FACETS']);
          sat2 := transposesat(sat, nops(data['FACETS']))
      fi;

#      eq := `&?`(assigned(d) and n = d+nops(data['EQUATIONS']),
#        data['EQUATIONS'], gausselim(data['EQUATIONS']));

      userinfo(3, POLYHEDRON, '`description of polyhedron is complete`');

      P := NEWCONE(n, [], data['VERTICES'], sat,
                   gausselim(data['AFFINE_HULL'], data['FACETS']), sat2)
  elif assigned(data['VERTICES'])
    then
      userinfo(3, POLYHEDRON, '`constructing polyhedron from VERTICES`');
      P := Hull(n, [], [], data['VERTICES'])
  elif assigned(data['FACETS']) and assigned(data['AFFINE_HULL'])
    then
      userinfo(3, POLYHEDRON,
        '`constructing polyhedron from FACETS and AFFINE_HULL`');
      P := Intersection(n, [], data['AFFINE_HULL'], data['FACETS'])
  elif assigned(data['POINTS'])
    then
      userinfo(3, POLYHEDRON, '`constructing polyhedron from POINTS`');
      P := Hull(n, [], [], data['POINTS'])
  elif assigned(data['INEQUALITIES']) and assigned(data['AFFINE_HULL'])
    then
      userinfo(3, POLYHEDRON,
        '`constructing polyhedron from INEQUALITIES and AFFINE_HULL`');
      P := Intersection(n, [], data['AFFINE_HULL'], data['INEQUALITIES'])
    else
     ERRORconstruct
  fi;
  NEWPOLYHEDRON(P)
end;
